<?php

namespace App\Models;

use App\Mail\Notification;
use Illuminate\Auth\Authenticatable;
use Illuminate\Notifications\Notifiable;
use Illuminate\Contracts\Auth\Access\Authorizable as AuthorizableContract;
use Illuminate\Contracts\Auth\Authenticatable as AuthenticatableContract;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\Hash;
use Laravel\Lumen\Auth\Authorizable;
use Illuminate\Auth\Passwords\CanResetPassword;
use Illuminate\Contracts\Auth\CanResetPassword as CanResetPasswordContract;
use Illuminate\Support\Facades\Mail;
use Spatie\Permission\Traits\HasRoles;
use Tymon\JWTAuth\Contracts\JWTSubject;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;



class User extends Model implements AuthenticatableContract, AuthorizableContract, JWTSubject, CanResetPasswordContract
{

    public const USER_CREATE_ACTIVITY = 'User Created';
    public const USER_UPDATE_ACTIVITY = 'User Updated';
    public const USER_DELETE_ACTIVITY = 'User Deleted';
    public const USER_PATH = '/user/list';

    use Authenticatable, Authorizable, HasFactory, CanResetPassword, HasRoles;

    /**
     * The attributes that are mass assignable.
     *
     * @var array
     */
    protected $fillable = [
        'unique_identity', 'first_name', 'last_name', 'gender', 'email', 'password', 'profile', 'birth_date', 'mobile', 'role_id', 'company_name', 'company_contact_person', 'account_number', 'ifsc_code', 'branch_name', 'tin_no', 'cst_no', 'pan_no', 'phone', 'address', 'city', 'state', 'pincode', 'bank_name', 'swift_code', 'branch_code', 'international_bank_code', 'national_bank_code', 'gst'
    ];

    /**
     * The attributes excluded from the model's JSON form.
     *
     * @var array
     */
    protected $hidden = [
        'password',
    ];

    /**
     * Retrieve the identifier for the JWT key.
     *
     * @return mixed
     */
    public function getJWTIdentifier()
    {
        return $this->getKey();
    }

    /**
     * Return a key value array, containing any custom claims to be added to the JWT.
     *
     * @return array
     */
    public function getJWTCustomClaims()
    {
        return [];
    }


    protected static function booted()
    {
        static::creating(function ($user) {
            if ($user->password != null) {
                $user->password = Hash::make($user->password);
            }
            $user->unique_identity = $user->generateUniqueIdentity(request()->role);
        });

        static::updating(function ($user) {
            if ($user->isDirty('password')) {
                $user->password = Hash::make($user->password);
            }
        });
    }

    public function getFullNameAttribute()
    {
        return $this->first_name . ' ' . $this->last_name;
    }

    public function customerContacts()
    {
        return $this->hasMany(CustomerContact::class);
    }


    public function getRole()
    {
        $roles = $this->getRoleNames();
        return $roles->isNotEmpty() ? $roles[0] : null;
    }

    public function generateUniqueIdentity($role)
    {
        $data = $this->select(['unique_identity'])->orderBy('created_at', 'desc')->first();

        if ($role) {
            if (!empty($data->unique_identity)) {
                $prefix = 'A-';
                switch ($role) {
                    case "customer":
                        $prefix = "C-";
                        break;
                    case "employee":
                        $prefix = "E-";
                        break;
                    case "supplier":
                        $prefix = "S-";
                        break;
                    default:
                        $prefix = "A-";
                }

                $split = explode("-", $data->unique_identity);
                $find = sizeof($split) - 1;
                $last_id = $split[$find];
                $number = intval($last_id) + 1;
                $new_no = sprintf('%06d', $number);
                $uniqID = $prefix . $new_no;
                return $uniqID;
            } else {
                $prefix = 'A-';
                switch ($role) {
                    case "customer":
                        $prefix = "C-";
                        break;
                    case "employee":
                        $prefix = "E-";
                        break;
                    case "supplier":
                        $prefix = "S-";
                        break;
                    default:
                        $prefix = "A-";
                }

                return $prefix . '000001';
            }
        }
    }

    public function sendPasswordResetNotification($token)
    {
        $systemName = getSetting('system_name');
        // Get email template content using helper
        $emailTemplate = getEmailContent('reset_password');
        $link = getHost() . '/#/session/reset-password/' . $token . '/' . $this->email;
        $logo = getSetting('logo');
        $logo_image = url('storage/' . $logo);
        $footer_image = url('storage/email_image/footer.png');
        $support_image = url('storage/email_image/support.png');
        $service_image = url('storage/email_image/service.png');
        $history_image = url('storage/email_image/history.png');
        $mailformat = $emailTemplate->body;
        $serch = array('{ name }', '{ link }', '{ systemname }');
        $replace = array($this->email, '<a href="http://' . $link . '">Click here</a>', $systemName);
        $message_content = str_replace($serch, $replace, $mailformat);
        $mailData = [
            'token' => $token,
            'email' => $this->email,
            'host' => getHost(),
            // 'host' => '192.168.1.131/amc_backend',
            'email_template' => 'ResetPassword',
            'email_subject' => $emailTemplate->subject,
            'content' => $message_content,
            'logo_image' => $logo_image,
            'footer_image' => $footer_image,
            'support_image' => $support_image,
            'service_image' => $service_image,
            'history_image' => $history_image
        ];
        // dd($token);
        Mail::to($this->email)->send(new Notification($mailData));
    }

    public function purchases()
    {
        return $this->hasMany(Purchase::class, 'supplier_id');
    }

    public function sales()
    {
        return $this->hasMany(Sale::class, 'customer_id');
    }

    public function complaints()
    {
        return $this->hasMany(Complaint::class, 'customer_id');
    }

    public function complaintsAssigned()
    {
        return $this->hasMany(Complaint::class, 'assign_to_id');
    }

    public function amcServicesAssigned()
    {
        return $this->hasMany(AmcService::class, 'assign_to');
    }

    public function quotations()
    {
        return $this->hasMany(Quotation::class, 'customer_id');
    }

    public function users()
    {
        return $this->belongsToMany(User::class);
    }

    public function conversations()
    {
        return $this->belongsToMany(Conversation::class, 'conversation_members', 'user_id', 'conversation_id');
    }

    public function sentConversations()
    {
        return $this->belongsToMany(Conversation::class, 'conversation_members', 'user_id', 'conversation_id')
        ->where('conversations.created_by', $this->id);
    }

    public function inboxConversations()
    {
        return $this->belongsToMany(Conversation::class, 'conversation_members', 'user_id', 'conversation_id')
            ->where('conversations.created_by', '!=', $this->id);
    } 
}
