<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model as Eloquent;

class Sale extends Eloquent
{
    public const SELL_CREATE_ACTIVITY = 'Sell Created';
    public const SELL_UPDATE_ACTIVITY = 'Sell Updated';
    public const SELL_DELETE_ACTIVITY = 'Sell Deleted';
    public const SELL_PATH = '/sales/list';

    /**
     * The attributes that are mass assignable.
     *
     * @var array
     */
    protected $fillable = [
        'bill_no',
        'customer_id',
        'date',
        'status_id',
        'discount',
        'sub_total',
        'tax_amount',
        'discount_amount',
        'gross_total',
        'paid_amount',
        'due_amount'
    ];

    protected $dates = [
        'date' => 'date:Y/m/d',
    ];

    protected static function booted()
    {
        static::creating(function ($sale) {
            $sale->bill_no = $sale->generateUniqueIdentity();
            $sale->date = date("Y-m-d", strtotime(request()->date));
        });
    }

    public function status()
    {
        return $this->belongsTo(Category::class, 'status_id');
    }

    public function customer()
    {
        return $this->belongsTo(User::class, 'customer_id');
    }

    public function productDetails()
    {
        return $this->hasMany(SaleProducts::class);
    }

    public function taxes()
    {
        return $this->belongsToMany(Tax::class, null, 'reference_ids', 'tax_ids');
    }

    public function generateUniqueIdentity()
    {
        $data = $this->select(['bill_no'])->orderBy('created_at', 'desc')->first();
        $prefix = 'S-';
        if (!empty($data->bill_no)) {


            $split = explode("-", $data->bill_no);
            $find = sizeof($split) - 1;
            $last_id = $split[$find];
            $number = intval($last_id) + 1;
            $new_no = sprintf('%06d', $number);
            $uniqID = $prefix . $new_no;
            return $uniqID;
        } else {
            return $prefix . '000001';
        }
    }

    public function paymentHistories()
    {
        return $this->hasMany(SalePaymentHistory::class);
    }

    public function updatePaymentStatus()
    {
        $this->paid_amount = $this->paymentHistories()->sum('amount');
        $this->due_amount = $this->gross_total - $this->paid_amount;

        if ($this->due_amount <= 0) {
            $this->status_id = 25; // Paid
        } elseif ($this->paid_amount > 0) {
            $this->status_id = 26; // Partial Paid
        } else {
            $this->status_id = 27; // Unpaid
        }

        $this->save();
    }
}
