<?php

namespace App\Http\Controllers;

use App\Events\SendFcmNotification;
use App\Models\Tax;
use App\Models\ActivityLog;
use App\Models\User;
use App\Models\Quotation;
use App\Models\Notifications;
use Illuminate\Http\Request;
use Barryvdh\DomPDF\Facade\Pdf;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Validator;
use App\Http\Resources\QuotationResource;
use App\Models\Setting;
use App\Mail\QuotationInvoice;
use Illuminate\Support\Facades\Storage;

class QuotationController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {
        try {
            if (auth()->user()->getRole() == 'admin' || auth()->user()->hasPermissionTo('quotation_view')) {
                if (auth()->user()->hasPermissionTo('quotation_own_data')) {
                    if (auth()->user()->getRole() == 'customer') {
                        $quotation = Quotation::where(['customer_id' => auth()->user()->id])->Orderby('created_at', 'desc')->get();
                    }
                } else {
                    $quotation = Quotation::Orderby('created_at', 'desc')->get();
                }

                return QuotationResource::collection($quotation)->additional(["error" => false, "message" => '']);
            } else {
                $data = [
                    'error' => true,
                    'message' => 'Unauthorized',
                    'error_code' => 401
                ];
                return response()->json($data, 401);
            }
        } catch (UserNotDefinedException $e) {
            $this->setResponse(true, $e->getMessage());
            return response()->json($this->_response, 500);
        }
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        //
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'customer_id' => 'required|exists:users,id',
            'date' => 'required|date',
            'status_id' => 'required|exists:categories,id',
            'details' => 'nullable',
            'product_id' => 'required|array',
            'product_id.*' => 'required|exists:products,id',
            'quantity' => 'required|array',
            'quantity.*' => 'required|numeric',
            'rate' => 'required|array',
            'rate.*' => 'required|numeric',
            'amount' => 'required|array',
            'amount.*' => 'required|numeric',
            'taxes' => 'filled|array',
            'taxes.*' => 'required|exists:taxes,id',
        ]);

        if ($validator->fails()) {
            $this->setResponse(true,  $validator->errors()->all());
            return response()->json($this->_response, 400);
        }

        try {
            if (auth()->user()->getRole() == 'admin' || auth()->user()->hasPermissionTo('quotation_add')) {

                $quotation = Quotation::create($request->all());

                $amount = 0;

                // Quotation Selected Products
                foreach ($request->product_id as $key => $value) {
                    $quotationProduct = $quotation->productDetails()->create([
                        'product_id' => $request->product_id[$key],
                        'quantity' => (int)$request->quantity[$key],
                        'rate' => $request->rate[$key],
                        'amount' => $request->amount[$key],
                    ]);
                    $amount += $request->amount[$key];
                }

                $totalTax = 0;
                $addedTaxes = [];
                // Tax calculation
                foreach ($request->taxes as $tax_id) {
                    $taxRow = Tax::find($tax_id);
                    $taxAmount = ($taxRow->percentage / 100) * $amount;
                    $totalTax += $taxAmount;
                    $addedTaxes[] = [
                        "label" => $taxRow->label,
                        "percentage" => $taxRow->percentage,
                        "tax_amount" => $taxAmount,
                    ];
                }

                $quotation->sub_total = $amount;
                $quotation->tax_amount = $totalTax;
                $quotation->gross_total = $amount + $totalTax;
                $quotation->save();

                $quotation->taxes()->attach($request->taxes);

                // Quotation mail to customer

                $settingData = Setting::get();
                $settingData = $settingData->first();
                $systemName = $settingData->system_name;
                // Get email template content using helper
                $emailTemplate = getEmailContent('quotation');
                $logo = getSetting('logo');
                $logo_image = url('storage/' . $logo);
                $footer_image = url('storage/email_image/footer.png');
                $support_image = url('storage/email_image/support.png');
                $service_image = url('storage/email_image/service.png');
                $history_image = url('storage/email_image/history.png');

                if ($emailTemplate) {
                    $pdfName = "quotation-" . $quotation->id . ".pdf";
                    // Pdf::loadView('pdf.Quotation', ['data'=>$quotation,'setting'=>$settingData])->save(public_path().'/invoice/'.$pdfName)->stream($pdfName);

                    $quotationInvoice = Pdf::loadView('pdf.Quotation', ['data' => $quotation, 'setting' => $settingData, 'taxes' => $addedTaxes]);
                    $content = $quotationInvoice->download()->getOriginalContent();
                    Storage::disk("public")->put('invoice/' . $pdfName, $content);

                    $mailformat = $emailTemplate->body;
                    $serch = array('{ username }', '{ systemname }');
                    $replace = array($quotation->customer->full_name, $systemName);
                    $message_content = str_replace($serch, $replace, $mailformat);

                    // Welcome Mail
                    $mailData = [
                        'host' => getHost(),
                        'subject' => $emailTemplate->subject,
                        'content' => $message_content,
                        'attachementPath' => "invoice/{$pdfName}",  //$quotationInvoice->output()
                        'logo_image' => $logo_image,
                        'footer_image' => $footer_image,
                        'support_image' => $support_image,
                        'service_image' => $service_image,
                        'history_image' => $history_image
                    ];

                    Mail::to($quotation->customer->email)->send(new QuotationInvoice($mailData));
                }

                $sourceName = auth()->id();
                    $user = User::find($sourceName);
                    $sourceId = $user ? $user->first_name : 'Unknown User';
                    $targetName = $quotation->customer->full_name;
                    $activityDetails = "Quation for customer {$targetName} created";

                    $data = [
                        'updated_by' => $sourceId,
                        'activity' => Quotation::QUOTATION_CREATE_ACTIVITY,
                        'activity_slug' =>Str::of(Quotation::QUOTATION_CREATE_ACTIVITY)->slug('_'),
                        'target_name' => $targetName,
                        'details' => $activityDetails,
                    ];
                    ActivityLog::log($data);
                    $notifications = [
                        [
                            'user_id' => $quotation->customer_id,
                            'target_id' => $quotation->id,
                            'updated_by' => $sourceId,
                            'content' => "New quotation recieved.",
                            'is_read' => 0,
                            'path' => Quotation::QUOTATION_PATH
                        ],
                    ];
                    Notifications::send($notifications);
                    $notificationData = [
                        [
                            'deviceToken' => $quotation->customer->device_token,
                            'title' => "New quotation recieved.",
                            'body' => "New quotation recieved. Check quotation details.",
                        ]
                        ];
                        event(new SendFcmNotification($notificationData));
                $this->_response['data'] = '';
                $this->setResponse(false, 'Quotation created successfully.');
                return response()->json($this->_response);
            } else {
                $data = [
                    'error' => true,
                    'message' => 'Unauthorized',
                    'error_code' => 401
                ];
                return response()->json($data, 401);
            }
        } catch (\Exception $e) {
            $this->setResponse(true,  $e->getMessage());
            return response()->json($this->_response, 500);
        }
    }

    /**
     * Display the specified resource.
     *
     * @param  \App\Models\Quotation  $quotation
     * @return \Illuminate\Http\Response
     */
    public function show(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'id' => 'required|exists:quotations,id',
        ]);

        if ($validator->fails()) {
            $this->setResponse(true,  $validator->errors()->all());
            return response()->json($this->_response, 400);
        }

        try {
            if (auth()->user()->getRole() == 'admin' || auth()->user()->hasPermissionTo('quotation_view')) {

                $quotation = Quotation::find($request->id);

                return (new QuotationResource($quotation))->additional(["error" => false, "message" => '']);
            } else {
                $data = [
                    'error' => true,
                    'message' => 'Unauthorized',
                    'error_code' => 401
                ];
                return response()->json($data, 401);
            }
        } catch (\Exception $e) {
            $this->setResponse(true,  $e->getMessage());
            return response()->json($this->_response, 500);
        }
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  \App\Models\Quotation  $quotation
     * @return \Illuminate\Http\Response
     */
    public function edit(Quotation $quotation)
    {
        //
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \App\Models\Quotation  $quotation
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, Quotation $quotation)
    {
        $validator = Validator::make($request->all(), [
            'id' => 'required|exists:quotations,id',
            'customer_id' => 'required|exists:users,id',
            'date' => 'required|date',
            'status_id' => 'required|exists:categories,id',
            'details' => 'nullable',
            'product_id' => 'required|array',
            'product_id.*' => 'required|exists:products,id',
            'quantity' => 'required|array',
            'quantity.*' => 'required|numeric',
            'rate' => 'required|array',
            'rate.*' => 'required|numeric',
            'amount' => 'required|array',
            'amount.*' => 'required|numeric',
            'taxes' => 'filled|array',
            'taxes.*' => 'required|exists:taxes,id',
        ]);

        if ($validator->fails()) {
            $this->setResponse(true,  $validator->errors()->all());
            return response()->json($this->_response, 400);
        }
        if (auth()->user()->getRole() == 'admin' || auth()->user()->hasPermissionTo('quotation_edit')) {

            $quotation = Quotation::find($request->id);

            $quotation->update($request->all());
            $amount = 0;

            $quotation->productDetails()->delete();

            // Quotation Selected Products
            foreach ($request->product_id as $key => $value) {
                $quotationProduct = $quotation->productDetails()->create([
                    'product_id' => $request->product_id[$key],
                    'quantity' => (int)$request->quantity[$key],
                    'rate' => $request->rate[$key],
                    'amount' => $request->amount[$key],
                ]);
                $amount += $request->amount[$key];
            }

            $totalTax = 0;
            // Tax calculation
            $addedTaxes = [];
            foreach ($request->taxes as $tax_id) {
                $taxRow = Tax::find($tax_id);
                $taxAmount = ($taxRow->percentage / 100) * $amount;
                $totalTax += $taxAmount;
                $addedTaxes[] = [
                    "label" => $taxRow->label,
                    "percentage" => $taxRow->percentage,
                    "tax_amount" => $taxAmount,
                ];
            }

            $quotation->sub_total = $amount;
            $quotation->tax_amount = $totalTax;
            $quotation->gross_total = $amount + $totalTax;
            $quotation->save();
            // dd($quotation->taxe_ids);
            // $quotation->taxes()->detach($quotation->tax_ids);

            $quotation->taxes()->sync($request->taxes);

            $pdfName = "quotation-" . $quotation->id . ".pdf";
            $settingData = Setting::get();
            $settingData = $settingData->first();

            $quotationInvoice = Pdf::loadView('pdf.Quotation', ['data' => $quotation, 'setting' => $settingData, 'taxes' => $addedTaxes]);
            $content = $quotationInvoice->download()->getOriginalContent();

            if (Storage::disk('public')->exists("invoice/{$pdfName}")) {
                Storage::disk('public')->delete("invoice/{$pdfName}");
            }

            Storage::disk("public")->put('invoice/' . $pdfName, $content);

                    $sourceName = auth()->id();
                    $user = User::find($sourceName);
                    $sourceId = $user ? $user->first_name : 'Unknown User';
                    $targetName = $quotation->customer->full_name;
                    $activityDetails = "Quation for customer {$targetName} updated";

                    $data = [
                        'updated_by' => $sourceId,
                        'activity' => Quotation::QUOTATION_UPDATE_ACTIVITY,
                        'activity_slug' =>Str::of(Quotation::QUOTATION_UPDATE_ACTIVITY)->slug('_'),
                        'target_name' => $targetName,
                        'details' => $activityDetails,
                    ];
                    ActivityLog::log($data);

            $this->_response['data'] = '';
            $this->setResponse(false, 'Quotation updated successfully.');
            return response()->json($this->_response);
        } else {
            $data = [
                'error' => true,
                'message' => 'Unauthorized',
                'error_code' => 401
            ];
            return response()->json($data, 401);
        }
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  \App\Models\Product  $product
     * @return \Illuminate\Http\Response
     */
    public function delete(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'id' => 'required|exists:quotations,id',
        ]);

        if ($validator->fails()) {
            $this->setResponse(true, $validator->errors()->all());
            return response()->json($this->_response, 400);
        }

        try {
            if (auth()->user()->getRole() == 'admin' || auth()->user()->hasPermissionTo('quotation_delete')) {

                $quotation =  Quotation::find($request->id);
                if ($quotation) {
                    $quotation->productDetails()->delete();
                    $quotation->delete();

                    $sourceName = auth()->id();
                    $user = User::find($sourceName);
                    $sourceId = $user ? $user->first_name : 'Unknown User';
                    $targetName = $quotation->customer->full_name;
                    $activityDetails = "Quation for customer {$targetName} deleted";

                    $data = [
                        'updated_by' => $sourceId,
                        'activity' => Quotation::QUOTATION_DELETE_ACTIVITY,
                        'activity_slug' =>Str::of(Quotation::QUOTATION_DELETE_ACTIVITY)->slug('_'),
                        'target_name' => $targetName,
                        'details' => $activityDetails,
                    ];
                    ActivityLog::log($data);

                    $this->setResponse(false, 'Quotation deleted successfully.');
                    return response()->json($this->_response);
                }
            } else {
                $data = [
                    'error' => true,
                    'message' => 'Unauthorized',
                    'error_code' => 401
                ];
                return response()->json($data, 401);
            }
        } catch (\Exception $e) {
            $this->setResponse(true, $e->getMessage());
            return response()->json($this->_response, 500);
        }
    }

    /*Delete multiple AMC Data*/
    public function deleteMultiQuotation(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'quotation_ids' => 'required|array',
            'quotation_ids.*' => 'required|exists:quotations,id',
        ]);

        if ($validator->fails()) {
            $this->setResponse(true, $validator->errors()->all());
            return response()->json($this->_response, 400);
        }
        try {
            if (auth()->user()->getRole() == 'admin' || auth()->user()->hasPermissionTo('quotation_delete')) {

                foreach ($request->quotation_ids as $id) {
                    $quotation =  Quotation::find($id);
                    if ($quotation) {
                        $quotation->productDetails()->delete();
                        $quotation->delete();
                    }
                }

                    $sourceName = auth()->id();
                    $user = User::find($sourceName);
                    $sourceId = $user ? $user->first_name : 'Unknown User';
                    $targetName = $quotation->customer->full_name;
                    $activityDetails = "Quation for customer {$targetName} deleted";

                    $data = [
                        'updated_by' => $sourceId,
                        'activity' => Quotation::QUOTATION_DELETE_ACTIVITY,
                        'activity_slug' =>Str::of(Quotation::QUOTATION_DELETE_ACTIVITY)->slug('_'),
                        'target_name' => $targetName,
                        'details' => $activityDetails,
                    ];
                    ActivityLog::log($data);

                $this->setResponse(false, 'Quotation deleted successfully.');
                return response()->json($this->_response);
            } else {
                $data = [
                    'error' => true,
                    'message' => 'Unauthorized',
                    'error_code' => 401
                ];
                return response()->json($data, 401);
            }
        } catch (\Exception $e) {
            $this->setResponse(true, $e->getMessage());
            return response()->json($this->_response, 500);
        }
    }
}
