<?php

namespace App\Http\Controllers;

use App\Models\Product;
use App\Models\ActivityLog;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Validator;
use App\Http\Resources\ProductResource;
use App\Http\Resources\dropdownResource;

class ProductController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {
        try {
            if (auth()->user()->getRole() == 'admin' || auth()->user()->hasPermissionTo('product_view')) {
                $products = Product::where(['is_archive' => 0])->Orderby('created_at', 'desc')->get();
                return ProductResource::collection($products)->additional(["error" => false, "message" => '']);
            } else {
                $data = [
                    'error' => true,
                    'message' => 'Unauthorized',
                    'error_code' => 401
                ];
                return response()->json($data, 401);
            }
        } catch (UserNotDefinedException $e) {
            $this->setResponse(true, $e->getMessage());
            return response()->json($this->_response, 500);
        }
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'name' => 'required',
            'model_number' => 'required',
            'brand_id' => 'required|exists:categories,id',
            'category_id' => 'required|exists:categories,id',
            'price' => 'required|numeric|gt:0',
            'unit_id' => 'required|exists:categories,id',
            'image' => 'filled|mimes:jpg,bmp,png,jpeg,svg|max:2048',
        ]);

        if ($validator->fails()) {
            $this->setResponse(true,  $validator->errors()->all());
            return response()->json($this->_response, 400);
        }

        try {
            if (auth()->user()->getRole() == 'admin' || auth()->user()->hasPermissionTo('product_add')) {

                $product = Product::create($request->except('image'));


                if ($request->has('image')) {

                    $image = $this->uploadFile($request->image, 'images/product');
                    if ($image != false) {
                        $product->image = $image;
                    }
                }

                $product->save();

                $sourceName = auth()->id();
                    $user = User::find($sourceName);
                    $sourceId = $user ? $user->first_name : 'Unknown User';
                $targetName = $product->name; // Assuming the product name is the target
                $data = [
                    'updated_by' => $sourceId,
                    'activity' => Product::PRODUCT_CREATE_ACTIVITY,
                    'activity_slug' =>Str::of(Product::PRODUCT_CREATE_ACTIVITY)->slug('_'),
                    'target_name' => $targetName,
                    'details' => "Product '{$targetName}' created" ,
                ];
                ActivityLog::log($data);
                $this->_response['data'] = '';
                $this->setResponse(false, 'Product created successfully.');
                return response()->json($this->_response);
            } else {
                $data = [
                    'error' => true,
                    'message' => 'Unauthorized',
                    'error_code' => 401
                ];
                return response()->json($data, 401);
            }
        } catch (\Exception $e) {
            $this->setResponse(true,  $e->getMessage());
            return response()->json($this->_response, 500);
        }
    }

    /**
     * Display the specified resource.
     *
     * @param  \App\Models\Product  $product
     * @return \Illuminate\Http\Response
     */
    public function show(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'id' => 'required|exists:products,id'
        ]);

        if ($validator->fails()) {
            $this->setResponse(true,  $validator->errors()->all());
            return response()->json($this->_response, 400);
        }

        try {
            if (auth()->user()->getRole() == 'admin' || auth()->user()->hasPermissionTo('product_view')) {

                $product = Product::find($request->id);

                return (new ProductResource($product))->additional(["error" => false, "message" => '']);
            } else {
                $data = [
                    'error' => true,
                    'message' => 'Unauthorized',
                    'error_code' => 401
                ];
                return response()->json($data, 401);
            }
        } catch (\Exception $e) {
            $this->setResponse(true,  $e->getMessage());
            return response()->json($this->_response, 500);
        }
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \App\Models\Product  $product
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'id' => 'required|exists:products,id',
            'name' => 'required',
            'model_number' => 'required',
            'brand_id' => 'required|exists:categories,id',
            'category_id' => 'required|exists:categories,id',
            'price' => 'required|numeric|gt:0',
            'unit_id' => 'required|exists:categories,id',
            'image' => 'filled|mimes:jpg,bmp,png,jpeg,svg|max:2048',
        ]);

        if ($validator->fails()) {
            $this->setResponse(true,  $validator->errors()->all());
            return response()->json($this->_response, 400);
        }

        try {
            if (auth()->user()->getRole() == 'admin' || auth()->user()->hasPermissionTo('product_edit')) {

                $product = Product::find($request->id);
                $product->update($request->except(['image']));


                if ($request->has('image')) {

                    $image = $this->uploadFile($request->image, 'images/product');
                    if ($image != false) {
                        $product->image = $image;
                    }
                }

                $product->save();
                $sourceName = auth()->id();
                    $user = User::find($sourceName);
                    $sourceId = $user ? $user->first_name : 'Unknown User';
            $targetName = $product->name;
            $data = [
                'updated_by' => $sourceId,
                'activity' => Product::PRODUCT_UPDATE_ACTIVITY,
                'activity_slug' =>Str::of(Product::PRODUCT_UPDATE_ACTIVITY)->slug('_'),
                'target_name' => $targetName,
                'details' => "Product '{$targetName}' updated",
            ];
            ActivityLog::log($data);

                return (new ProductResource($product))->additional(["error" => false, "message" => 'Product updated successfully.']);
            } else {
                $data = [
                    'error' => true,
                    'message' => 'Unauthorized',
                    'error_code' => 401
                ];
                return response()->json($data, 401);
            }
        } catch (\Exception $e) {
            $this->setResponse(true,  $e->getMessage());
            return response()->json($this->_response, 500);
        }
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  \App\Models\Product  $product
     * @return \Illuminate\Http\Response
     */
    public function delete(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'id' => 'required|exists:products,id',
        ]);

        if ($validator->fails()) {
            $this->setResponse(true, $validator->errors()->all());
            return response()->json($this->_response, 400);
        }

        try {
            if (auth()->user()->getRole() == 'admin' || auth()->user()->hasPermissionTo('product_delete')) {

                $product =  Product::find($request->id);
                if ($product) {
                    $totalAmcRecords = $product->amc->count();
                    $totalQuotationRecords = $product->amc->count();
                    $totalSaleRecords = $product->amc->count();
                    if ($totalAmcRecords || $totalQuotationRecords || $totalSaleRecords) {
                        $dependency = [
                            "Amc records" => $totalSaleRecords,
                            "Quotation records" => $totalQuotationRecords,
                            "Sale records" => $totalSaleRecords,
                        ];
                        $this->_response['data'] = $dependency;
                        $this->setResponse(false, "Product has dependancy so you can't delete it.");
                        return response()->json($this->_response, 200);
                    }
                    $product->delete();
                    $sourceName = auth()->id();
                    $user = User::find($sourceName);
                    $sourceId = $user ? $user->first_name : 'Unknown User';
                    $targetName = $product->name; // Assuming the target is the product name
                    $data = [
                        'updated_by' => $sourceId,
                        'activity' => Product::PRODUCT_DELETE_ACTIVITY,
                        'activity_slug' =>Str::of(Product::PRODUCT_DELETE_ACTIVITY)->slug('_'),
                        'target_name' => $targetName,
                        'details' => "Product '{$targetName}' deleted",
                    ];
                    ActivityLog::log($data);

                    $this->setResponse(false, 'Product deleted successfully.');
                    return response()->json($this->_response);
                }
            } else {
                $data = [
                    'error' => true,
                    'message' => 'Unauthorized',
                    'error_code' => 401
                ];
                return response()->json($data, 401);
            }
        } catch (\Exception $e) {
            $this->setResponse(true, $e->getMessage());
            return response()->json($this->_response, 500);
        }
    }

    /*Delete multiple AMC Data*/
    public function deleteMultiProduct(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'product_ids' => 'required|array',
            'product_ids.*' => 'required|exists:products,id',
        ]);

        if ($validator->fails()) {
            $this->setResponse(true, $validator->errors()->all());
            return response()->json($this->_response, 400);
        }
        try {
            if (auth()->user()->getRole() == 'admin' || auth()->user()->hasPermissionTo('product_delete')) {
                foreach ($request->product_ids as $id) {
                    $product =  Product::find($id);
                    if ($product) {
                        if ($product->amc->count() || $product->quotation->count() || $product->sale->count()) {
                            $this->setResponse(false, "Some product has dependancy so you can't delete it.");
                        } else {
                            $sourceName = auth()->id();
                    $user = User::find($sourceName);
                    $sourceId = $user ? $user->first_name : 'Unknown User';
                            $targetName = $product->name; // Assuming you want to log the product name as the target
                            $activity = Product::PRODUCT_DELETE_ACTIVITY;
                            $details = $targetName . ' was deleted.';

                            $data = [
                                'updated_by' => $sourceId,
                                'activity' => $activity,
                                'activity_slug' =>Str::of($activity)->slug('_'),
                                'target_name' => $targetName,
                                'details' => $details,
                            ];

                        ActivityLog::log($data);
                            $product->delete();
                            $this->setResponse(false, 'Products deleted successfully.');
                        }
                    }
                }

                return response()->json($this->_response);
            } else {
                $data = [
                    'error' => true,
                    'message' => 'Unauthorized',
                    'error_code' => 401
                ];
                return response()->json($data, 401);
            }
        } catch (\Exception $e) {
            $this->setResponse(true, $e->getMessage());
            return response()->json($this->_response, 500);
        }
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \App\Models\Product  $product
     * @return \Illuminate\Http\Response
     */
    public function makeArchive(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'id' => 'required|exists:products,id'
        ]);

        if ($validator->fails()) {
            $this->setResponse(true,  $validator->errors()->all());
            return response()->json($this->_response, 400);
        }

        try {
            $product = Product::find($request->id);
            if ($product->amc->count() || $product->quotation->count() || $product->sale->count()) {
                $this->setResponse(false, "Product has dependancy so you can't archive it.");
                return response()->json($this->_response, 200);
            }
            $product->update(['is_archive' => 1]);

            if ($product->stock) {
                $product->stock->delete();
            }


            return (new ProductResource($product))->additional(["error" => false, "message" => 'Product archived successfully.']);
        } catch (\Exception $e) {
            $this->setResponse(true,  $e->getMessage());
            return response()->json($this->_response, 500);
        }
    }

    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function archiveList()
    {
        try {
            $products = Product::where(['is_archive' => 1])->Orderby('updated_at', 'desc')->get();
            return ProductResource::collection($products)->additional(["error" => false, "message" => '']);
        } catch (UserNotDefinedException $e) {
            $this->setResponse(true, $e->getMessage());
            return response()->json($this->_response, 500);
        }
    }

    /*User List By Role */
    public function produtcDropdown()
    {
        try {
            $options = Product::where(['is_archive' => 0])->select(['id', 'name'])->get();
            // dd($options);
            return dropdownResource::collection($options)->additional(["error" => false, "message" => '']);
        } catch (UserNotDefinedException $e) {
            $this->setResponse(true, $e->getMessage());
            return response()->json($this->_response, 500);
        }
    }
}
