<?php

namespace App\Http\Controllers;


use App\Models\User;
use GuzzleHttp\Client;
use App\Models\License;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Schema;
use Illuminate\Support\Facades\Artisan;
use Illuminate\Support\Facades\Validator;
use GuzzleHttp\Exception\RequestException;

class InstallationController extends Controller
{
    /**
     * Store a new user.
     *
     * @param  Request  $request
     * @return Response
     */
    public function index(Request $request)
    {

        $whitelist = [
            // IPv4 address
            '127.0.0.1',
            'localhost',
            'localhost',
            // IPv6 address
            '::1'
        ];

        $domain_name = $request->domain_name;
        $licence_key = $request->purchase_key;
        $registered_email = $request->purchase_email;

        if (!in_array($domain_name, $whitelist)) {
            try {
                $api_server = 'license.dasinfomedia.com';
                $client = new Client();

                // Check if the server is up
                $client->get("http://{$api_server}", ['timeout' => 2]);

                $url = 'https://license.dasinfomedia.com/admin/api/license/register';
                $params = [
                    'pkey' => $licence_key,
                    'email' => $registered_email,
                    'domain' => $domain_name
                ];

                Log::info('Attempting license verification', ['domain' => $domain_name, 'email' => $registered_email]);

                $response = $client->post($url, [
                    'form_params' => $params
                ]);

                $result = json_decode($response->getBody()->getContents(), true);
                Log::info('License verification response received', ['result' => $result]);

                if ($result['error']) {
                    throw new \Exception($result['message']);
                }

                switch ($result['data']) {
                    case '0':
                        Log::info('License verification successful', ['domain' => $domain_name]);
                        // Proceed with the rest of your code for successful verification
                        break;
                    case '1':
                        throw new \Exception("Invalid license. Please enter a correct purchase key.");
                    case '2':
                        throw new \Exception("This purchase key is already registered with a different domain. Please contact us at sales@dasinfomedia.com");
                    case '3':
                        throw new \Exception("There was a problem registering your license. Please try again later or contact us at sales@dasinfomedia.com");
                    case '4':
                        Log::info('License already registered with the same domain', ['domain' => $domain_name]);
                        break;
                    default:
                        throw new \Exception("Unexpected response from license server. Please try again later.");
                }

            } catch (RequestException $e) {
                Log::error('License verification request failed', [
                    'domain' => $domain_name,
                    'error' => $e->getMessage()
                ]);
                $this->setResponse(true, "Connection problem occurred while verifying the license.");
                return response()->json($this->_response, 400);
            } catch (\Exception $e) {
                Log::error('License verification failed', [
                    'domain' => $domain_name,
                    'error' => $e->getMessage()
                ]);
                $this->setResponse(true, $e->getMessage());
                return response()->json($this->_response, 400);
            }
        }

        // MongoDB Connection
        $file = base_path('.env');
        $content = file_get_contents($file);
        $mySqlHost = $request->db_host;
        $mySqlPort = $request->db_port;
        $databaseName = $request->db_name;
        $mySqlUserName = $request->db_username;
        $mySqlPassword = $request->db_pass;

        try {
            $connection = new \PDO("mysql:host={$mySqlHost};port={$mySqlPort};dbname={$databaseName}", $mySqlUserName, $mySqlPassword);
        } catch (\PDOException $e) {
            $errorCode = $e->getCode();
            $errorMessage = '';

            if ($errorCode === 1045) {
                // Incorrect database credentials
                $errorMessage = "Database connection failed. Pelase check your database credentials.";
            } elseif ($errorCode === 1044) {
                // User does not have access to the database
                $errorMessage = "Database connection failed. User does not have access to the specified database.";
            } elseif ($errorCode === 1049) {
                // Unknown database
                $errorMessage = "Database '{$databaseName}' does not exist. Please create the database or check the database name.";
            } else {
                // Other PDO exceptions
                $errorMessage = "Database connection failed. Error: " . $e->getMessage();
            }

            $this->setResponse(true, $errorMessage);
            return response()->json($this->_response, 500);
            exit;
        }

        //Update db detail in .env
        $content = str_replace(["CUST_HOST", "CUST_PORT", "CUST_DB", "CUST_USERNAME", "CUST_PASS"], [$mySqlHost, $mySqlPort, $databaseName, $mySqlUserName, $mySqlPassword], $content);
        $status = file_put_contents($file, $content);

        //Run time db config
        config([
            'database.connections.mysql.database' => $request->db_name,
            'database.connections.mysql.username' => $request->db_username,
            'database.connections.mysql.password' => $request->db_pass,
            'database.connections.mysql.port' => $request->db_port,
            'database.connections.mysql.host' => $request->db_host,
        ]);

        DB::purge('mysql');

        DB::reconnect('mysql');

        Schema::connection('mysql')->getConnection()->reconnect();

        //Reconnect
        app()->forgetInstance('mysql');

        app()->instance('mysql', $request->db_name);

        Artisan::call('migrate');
        Artisan::call('db:seed --class=CategorySeeder');
        Artisan::call('db:seed --class=ModuleSeeder');
        Artisan::call('db:seed --class=PermissionSeed');
        Artisan::call('db:seed --class=RoleSeed');
        Artisan::call('db:seed --class=SettingSeeder');

        // dd(Artisan::output());

        //Create Admin User
        $admin = User::create([
            'first_name' => "Admin",
            'last_name' => "Admin",
            'gender' => "male",
            'email' => $request->email,
            'password' => $request->password,
            'birth_date' => "1994-01-01",
            'mobile' => "1234567891",
        ]);

        $admin->assignRole("admin");

        $this->setResponse(false, "Success.");
        return response()->json($this->_response, 200);
    }

    public function checkInstallation()
    {
        try {
            $userCount = User::count();
            $installStatus = $userCount > 0 ? true : false;

            $this->_response['data'] = ["install_status" => $installStatus];

            return response()->json($this->_response, 200);
        } catch (\Exception $e) {
            $this->_response['data'] = ["install_status" => false];
            return response()->json($this->_response, 200);
        }
    }

    public function saveLicenseData(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'app_email' => 'required',
            'app_url' => 'required',
            'app_license_key' => 'required',
        ]);

        if ($validator->fails()) {
            $this->setResponse(true, $validator->errors()->all());
            return response()->json($this->_response, 400);
        }

        $license = new License(
            [
                'app_email' => $request->app_email,
                'app_url' => $request->app_url,
                'app_license_key' => $request->app_license_key,
            ]
        );

        $license->save();

        return response()->json([
            'status' => true,
            "code" => 200,
            'message' => 'License Added Successfully',
        ], 200);
    }

    public function getLicenseData()
    {
        $license = License::latest()->first();

        if ($license) {
            return response()->json([
                'status' => true,
                'code' => 200,
                'message' => 'Get License Successfully',
                'data' => [
                    'app_email' => $license->app_email,
                    'app_url' => $license->app_url,
                    'app_license_key' => $license->app_license_key,
                ],
            ], 200);
        }

        return response()->json([
            'status' => false,
            'code' => 401,
            'message' => 'Record not found',
            'data' => null,
        ], 401);
    }
}
