<?php

namespace App\Http\Controllers;

use App\Http\Resources\ExpenseListResource;
use App\Http\Resources\ExpenseMonthlyReportResource;
use App\Models\Expense;
use App\Models\User;
use App\Models\ActivityLog;
use Illuminate\Http\Request;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Validator;

class ExpenseController extends Controller
{

    public function index()
    {
        if (auth()->user()->getRole() == 'admin' || auth()->user()->hasPermissionTo('expense_view')) {
            if (auth()->user()->hasPermissionTo('expense_own_data')) {
                $expenses = Expense::where(['customer_id' => auth()->user()->id])->get();
            } else {
                $expenses = Expense::all();
            }
            return ExpenseListResource::collection($expenses)->additional(["error" => false, "message" => '']);;
        } else {
            $data = [
                'error' => true,
                'message' => 'Unauthorized',
                'error_code' => 401
            ];
            return response()->json($data, 401);
        }
    }

    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'customer_id' => 'required|exists:users,id',
            'status_id' => 'required|exists:categories,id',
            'date' => 'required|date',
            'label' => 'required',
            'entries' => 'required|json'
        ]);

        if ($validator->fails()) {
            $this->setResponse(true, $validator->errors()->all());
            return response()->json($this->_response, 400);
        }
        if (auth()->user()->getRole() == 'admin' || auth()->user()->hasPermissionTo('expense_add')) {

            $expense = Expense::create([
                "customer_id" => $request->customer_id,
                "status_id" => $request->status_id,
                "date" => $request->date,
                "label" => $request->label,
                "entries" => $request->entries,
            ]);

            $sourceName = auth()->id();
            $users = User::find($sourceName);
            $sourceId = $users ? $users->first_name : 'Unknown User';
            $data = [
                'updated_by' => $sourceId,
                'activity' => Expense::EXPENSE_CREATE_ACTIVITY,
                'activity_slug' =>Str::of(Expense::EXPENSE_CREATE_ACTIVITY)->slug('_'),
                'target_name' => $expense->label,
                'details' => "Expense for user '{$expense->customer->full_name}' created",
            ];
            ActivityLog::log($data);
            $this->setResponse(false, 'Expense created successfully');
            return response()->json($this->_response, 200);
        } else {
            $data = [
                'error' => true,
                'message' => 'Unauthorized',
                'error_code' => 401
            ];
            return response()->json($data, 401);
        }
    }

    public function update(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'expense_id' => 'required|exists:expenses,id',
            'customer_id' => 'required|exists:users,id',
            'status_id' => 'required|exists:categories,id',
            'date' => 'required|date',
            'label' => 'required',
            'entries' => 'required|json'
        ]);

        if ($validator->fails()) {
            $this->setResponse(true, $validator->errors()->all());
            return response()->json($this->_response, 400);
        }

        if (auth()->user()->getRole() == 'admin' || auth()->user()->hasPermissionTo('expense_edit')) {

            $expense = Expense::find($request->expense_id);
            $expense->customer_id = $request->customer_id;
            $expense->status_id = $request->status_id;
            $expense->date = $request->date;
            $expense->label = $request->label;
            $expense->entries = $request->entries;
            $expense->save();
            $sourceName = auth()->id();
            $users = User::find($sourceName);
            $sourceId = $users ? $users->first_name : 'Unknown User';
            $data = [
                'updated_by' => $sourceId,
                'activity' => Expense::EXPENSE_UPDATE_ACTIVITY,
                'activity_slug' =>Str::of(Expense::EXPENSE_UPDATE_ACTIVITY)->slug('_'),
                'target_name' => $expense->label,
                'details' => "Expense for user '{$expense->customer->full_name}' updated",
            ];
            ActivityLog::log($data);
            $this->setResponse(false, 'Expense updated successfully');
            return response()->json($this->_response, 200);
        } else {
            $data = [
                'error' => true,
                'message' => 'Unauthorized',
                'error_code' => 401
            ];
            return response()->json($data, 401);
        }
    }

    public function destroy(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'expense_ids' => 'required|array',
            'expense_ids.*' => 'required|exists:expenses,id',
        ]);

        if ($validator->fails()) {
            $this->setResponse(true, $validator->errors()->all());
            return response()->json($this->_response, 400);
        }
        if (auth()->user()->getRole() == 'admin' || auth()->user()->hasPermissionTo('expense_delete')) {
            $deletedExpenses = Expense::whereIn('id', $request->expense_ids)->get();

            Expense::whereIn('id', $request->expense_ids)->delete();
            
            $sourceName = auth()->id();
            $users = User::find($sourceName);
            $sourceId = $users ? $users->first_name : 'Unknown User';
            foreach ($deletedExpenses as $expense) {
                $data = [
                    'updated_by' => $sourceId,
                    'activity' => Expense::EXPENSE_DELETE_ACTIVITY,
                    'activity_slug' =>Str::of(Expense::EXPENSE_DELETE_ACTIVITY)->slug('_'),
                    'target_name' => $expense->label,
                    'details' => "Expense for user '{$expense->customer->full_name}' deleted",
                ];
                ActivityLog::log($data);
            }
            $this->setResponse(false, 'Expense deleted successfully');
            return response()->json($this->_response, 200);
        } else {
            $data = [
                'error' => true,
                'message' => 'Unauthorized',
                'error_code' => 401
            ];
            return response()->json($data, 401);
        }
    }

    public function monthlyReport(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'start_date' => 'required|date',
            'end_date' => 'required|date',
        ]);

        if ($validator->fails()) {
            $this->setResponse(true, $validator->errors()->all());
            return response()->json($this->_response, 400);
        }
        if (auth()->user()->getRole() == 'admin' || auth()->user()->hasPermissionTo('expense_view')) {

            $startDate = date('Y-m-d', strtotime($request->start_date));
            $endDate = date('Y-m-d', strtotime($request->end_date));

            $expenseData = Expense::whereDate('date', '>=', $startDate)->whereDate('date', '<=', $endDate)->get();
            return ExpenseMonthlyReportResource::collection($expenseData)->additional(["error" => false, "message" => '']);
        } else {
            $data = [
                'error' => true,
                'message' => 'Unauthorized',
                'error_code' => 401
            ];
            return response()->json($data, 401);
        }
    }
}
